#! /usr/bin/env python
# encoding: utf-8

from __future__ import absolute_import

import os
import io
import requests.exceptions
import sys
from youtube_upload import main

class upload_in_chunks(object):
    def __init__(self, filename, chunksize=1 << 13, uploadparam=None):
        self.filename = filename
        self.chunksize = chunksize
        self.totalsize = os.path.getsize(filename)
        self.readsofar = 0
        self.prepercent = ''
        self.uploadparam = uploadparam

    def __iter__(self):
        with open(self.filename, 'rb') as file:
            while True:
                data = file.read(self.chunksize)
                if not data:
                    sys.stderr.write("\n")
                    break
                self.readsofar += len(data)
                percent = self.readsofar * 1e2 / self.totalsize
                # sys.stderr.write("\r{percent:3.0f}%".format(percent=percent))
                curpercent = "\r{percent:3.0f}%".format(percent=percent)
                if self.prepercent != curpercent:
                    # print curpercent
                    self.prepercent = curpercent
                    if  self.uploadparam != None:
                        if self.uploadparam.stop:
                            raise BaseException
                        if main.notifyEvent != None:
                            main.notifyEvent('PROGRESS={}'.format(int(percent)))
                        #self.uploadparam.SendMessage(1,str(int(percent)))
                yield data

    def __len__(self):
        return self.totalsize

class IterableToFileAdapter(object):
    def __init__(self, iterable):
        self.iterator = iter(iterable)
        self.length = len(iterable)

    def read(self, size=-1): # TBD: add buffer for `len(data) > size` case
        return next(self.iterator, b'')

    def __len__(self):
        return self.length

class UploadVideoMixin(object):
    """Handle uploading a new video to the Vimeo API."""

    UPLOAD_ENDPOINT = '/me/videos'
    REPLACE_ENDPOINT = '{video_uri}/files'

    def upload(self, filename, upgrade_to_1080=False, uploadparam=None):
        """Upload the named file to Vimeo."""

        if self.UPLOAD_ENDPOINT != None and main.notifyEvent != None and type(self.UPLOAD_ENDPOINT) == str:
           print('UPLOAD_ENDPOINT: %s' % self.UPLOAD_ENDPOINT)
        ticket = self.post(self.UPLOAD_ENDPOINT,
            data={'type': 'streaming',
                'upgrade_to_1080': 'true' if upgrade_to_1080 else 'false'})

        return self._perform_upload(filename, ticket, uploadparam)

    def replace(self, video_uri, filename, upgrade_to_1080=False):
        """Replace the video at the given uri with the named source file."""
        uri = self.REPLACE_ENDPOINT.format(video_uri=video_uri)

        ticket = self.put(uri,
            data={'type': 'streaming',
                'upgrade_to_1080': 'true' if upgrade_to_1080 else 'false'})

        return self._perform_upload(filename, ticket)

    def _perform_upload(self, filename, ticket, uploadparam):
        """Take an upload ticket and perform the actual upload."""

        print('ticket.status_code = %d' % ticket.status_code)
        assert ticket.status_code == 201, ticket.text

        ticket = ticket.json()

        # Perform the actual upload.
        target = ticket['upload_link']
        size = os.path.getsize(filename)
        last_byte = 0
        # with io.open(filename, 'rb') as f:
        it = upload_in_chunks(filename, 1024*1024, uploadparam)
        while last_byte < size:
            try:
                self._make_pass(target, IterableToFileAdapter(it), size, last_byte)
            except requests.exceptions.Timeout:
                    # If there is a timeout here, we are okay with it, since
                    # we'll check and resume.
                pass
            last_byte = self._get_progress(target, size)

        # Perform the finalization and get the location.
        finalized_resp = self.delete(ticket['complete_uri'])

        assert finalized_resp.status_code == 201, "Failed to create the video."

        return finalized_resp.headers.get('Location', None)

    def _get_progress(self, upload_target, filesize):
        """Test the completeness of the upload."""
        progress_response = self.put(
            upload_target,
            headers={'Content-Range': 'bytes */*'})

        range_recv = progress_response.headers.get('Range', None)
        _, last_byte = range_recv.split('-')

        return int(last_byte)

    def _make_pass(self, upload_target, f, size, last_byte):
        """Make a pass at uploading.

        This particular function may do many things.  If this is a large upload
        it may terminate without having completed the upload.  This can also
        occur if there are network issues or any other interruptions.  These
        can be recovered from by checking with the server to see how much it
        has and resuming the connection.
        """
        response = self.put(
            upload_target,
            timeout=None,
            headers={
                'Content-Length': size,
                'Content-Range': 'bytes: %d-%d/%d' % (last_byte, size, size)
            }, data=f)

        assert response.status_code == 200, response.text


class UploadPictureMixin(object):
    """Functionality for uploading a picture to Vimeo for another object
    (video, user, etc).
    """

    def upload_picture(self, obj, filename, activate=False):
        """Upload a picture for the object.

        The object (obj) can be the URI for the object or the response/parsed
        json for it.
        """
        if isinstance(obj, str):
            # TODO:  Add filtering down to just the picture connection field.
            obj = self.get(obj)
            assert obj.status_code == 200, "Failed to load the target object."
            obj = obj.json()

        # Get the picture object.
        picture = self.post(obj['metadata']['connections']['pictures']['uri'])

        assert picture.status_code == 201, \
            "Failed to create a new picture with Vimeo."

        picture = picture.json()

        with io.open(filename, 'rb') as f:
            upload_resp = self.put(picture['link'], data=f)
        assert upload_resp.status_code == 200, "Failed uploading"

        if activate:
            active = self.patch(picture['uri'], data={"active": "true"})
            assert active.status_code == 200, "Failed activating"
            picture['active'] = True

        return picture

class UploadTexttrackMixin(object):
    """Functionality for uploading a texttrack to Vimeo for a video.
    """
    TEXTTRACK_ENDPOINT = '{video_uri}/texttracks'

    def upload_texttrack(self, video_uri, track_type, language, filename):
        """Upload the texttrack at the given uri with the named source file."""
        uri = self.TEXTTRACK_ENDPOINT.format(video_uri=video_uri)
        name = filename.split('/')[-1]

        texttrack = self.post(uri,
                              data={'type': track_type,
                                    'language': language,
                                    'name': name})

        assert texttrack.status_code == 201, \
            "Failed to create a new texttrack with Vimeo."

        texttrack = texttrack.json()

        with io.open(filename, 'rb') as f:
            upload_resp = self.put(texttrack['link'], data=f)
        assert upload_resp.status_code == 200, "Failed uploading"

        return texttrack


class UploadMixin(UploadVideoMixin, UploadPictureMixin, UploadTexttrackMixin):
    """Handle uploading to the Vimeo API."""
    pass
